<?php

class Base_Image
{
	private $__acceptedMime = array(
		"png" => "image/png",
		"jpg" => "image/jpeg",
		"gif" => "image/gif"
	);
	
	protected $_tempfile;
	protected $_filename;
	public $_imagetype;
	public $_imagemime;
	protected $_image;
	
	private $_filetimestamp;
	
	/**
	 * Create a new image, optionally loading from a file
	 *
	 * @param string $filename
	 */
	public function __construct($filename = "")
	{
		$this->load($filename);
	}
	
	
	public function load($filename="")
	{
		if ($filename=="" or !file_exists($filename)) {
			return false;
		}
		
		$imginfo = @getimagesize($filename);
		if ($imginfo == false) {
			throw new Exception("Invalid file type for file " .$filename);
		}
		if ( ! in_array($imginfo["mime"], $this->__acceptedMime) ) {
			throw new Exception("Unrecognized file type " .$imginfo["mime"]. " (".$imginfo[2].")");
		}
		
		$this->_imagetype = $imginfo[2];
		$this->_imagemime = $imginfo["mime"];
		$this->_tempfile = $filename;
		
		switch($this->_imagetype) {
			case IMAGETYPE_GIF:
				$this->_image = imagecreatefromgif($filename);
			break;
			
			case IMAGETYPE_JPEG:
				$this->_image = imagecreatefromjpeg($filename);
			break;
			
			case IMAGETYPE_PNG:
				$this->_image = imagecreatefrompng($filename);
			break;
			
			default:
				throw new Exception("Invalid image type " .$this->_imagemime. " (" .$this->_imagetype. ")");
			break;
		}
		
		return $this;
	}	
	
	/**
	 * Save the original file to disk
	 *
	 * @param string $path
	 * @return self for chaining
	 * 
	 * Save the original file to disk in a specified path.
	 * Bugs known: it works only in the same "page" where the upload took place,
	 * as it gets the image from /tmp
	 * 
	 */
	public function save($fullname)
	{
		$path = dirname($fullname);

		if (!empty($this->_image)) {
			if ( ! file_exists($path) || ! is_dir($path)) {
				if ( ! @mkdir($path, 0755, true)) {
					throw new Exception("Unable to create image directory: " .$path);
				}
			}
			
			if (file_exists($fullname)) {
				if ( ! @unlink($fullname) ) {
					throw new Exception("Unable to delete old photo: " .$fullname);
				}
			}
			
			if ( ! @rename($this->_tempfile, $fullname) ) {
				throw new Exception("Unable to save image file: " .$fullname);
			}
			$this->_filename = $fullname;
			chmod($fullname, 0644);
		}
		
		return $this;
	}
	
	/**
	 * Create a thumbnail
	 *
	 * @param int $x New width
	 * @param int $y New height
	 * @param string $ratio Ratio settings
	 * @return thumbnail image in internal gd format
	 * 
	 * ratio:
	 * - keep : keeps image ratio within XxY bounding box
	 * - x    : keeps image ratio with width x
	 * - y    : keeps image ratio with height y
	 * - crop : crops image in the middle if it cannot keep ratio
	 * - no   : don't keep ratio, resize exactly XxY
	 */
	
	public function thumb($x=0, $y=0, $ratio="keep")
	{
		if ($x == 0 && $y == 0) {
			throw new Exception("Invalid image sizes requested");
		}
		
			// Forza il rapporto se e` stata specificata una sola dimensione
		$x==0 and $ratio="y";
		$y==0 and $ratio="x";
		
		
		$oldx = imagesx($this->_image);
		$oldy = imagesy($this->_image);
		
		$width = 0;
		$height = 0;
		$left = 0;
		$right = $oldx;
		$top = 0;
		$bottom = $oldy;
		
			// Calcola le dimensioni dell'immagine risultante
		switch ($ratio) {
			case "keep":
				$width = $x;
				$height = intval(($oldy * $width) / $oldx);
				
				if ($height > $y) {
					$height = $y;
					$width = intval(($oldx * $height) / $oldy);
				}
			break;
			
			case "x":
				$width = $x;
				$height = intval( ($oldy * $width) / $oldx);
			break;
			
			case "y":
				$height = $y;
				$width = intval( ($oldx * $height) / $oldy);
			break;
			
			case "crop":
				$width = $x;
				$height = $y;
				
				$xratio = $oldx / $x;
				$yratio = $oldy / $y;
				
				if ($xratio > $yratio) {
					$newwidth = $yratio * $x; 
					
					$left = intval( ($oldx/2) - ($newwidth/2) );
					$right = $left+$newwidth;

				} elseif ($xratio < $yratio) {
					$newheight = $xratio * $y;
					
					$top = intval( ($oldy/2) - ($newheight/2) );
					$bottom = $top+$newheight;
				}
			break;
			
			case "no":
				$width = $x;
				$height = $y;
			break;
			
			default:
				throw new Exception("Invalid image ratio requested");
			break;
		}

		$dstimg = imagecreatetruecolor($width, $height);
										//        dstx_y srcx   srcy  dstw     dsth     srcw    srch
		imagecopyresampled($dstimg, $this->_image, 0, 0, $left, $top, $width, $height, $right, $bottom);
		
		return $dstimg;
	}
}
